﻿using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using UnityEngine;

//PerlinNoiseGenerator is used as a wrapper class for implementing Perlin Noise functionality
public class PerlinNoiseGenerator
{
    public Vector2Int SamplesSize;  //Size of the samples to obtain
    public float Scale; //Scale between samples
    public Vector2 Offset;  //Offset into the noise function

    public List<List<Color>> Samples { get; private set; }  //The Perlin Noise sample values

    public PerlinNoiseGenerator(Vector2Int samplesSize, float scale, Vector2Int stepSize, Vector2 offset)
    {
        SamplesSize = samplesSize;
        Scale = scale;
        Offset = offset;
    }

    /// <summary>
    /// Generates the samples of Perlin Noise based on the set values
    /// </summary>
    public void Generate()
    {
        Samples = new List<List<Color>>();

        for(int x = 0; x < SamplesSize.x; x++)
        {
            Samples.Add(new List<Color>());
            for(int y = 0; y < SamplesSize.y; y++)
            {
                Samples[x].Add(GetSample(x, y));
            }
        }
    }

    /// <summary>
    /// Gets the colour sample value of Perlin Noise at the specified coordinate
    /// </summary>
    /// <param name="x">The x coordinate of the sample</param>
    /// <param name="y">The y coordinate of the sample</param>
    /// <returns>The sampled value</returns>
    private Color GetSample(int x, int y)
    {
        float xCoord = (float)x / SamplesSize.x * Scale + Offset.x;
        float yCoord = (float)y / SamplesSize.y * Scale + Offset.y;

        float sampleValue = Mathf.PerlinNoise(xCoord, yCoord);
        return new Color(sampleValue, sampleValue, sampleValue);
    }
}
